#!/usr/bin/perl
#
# qlnx-psets.pl by Davide Libenzi ( Linux patch-sets bits )
# Copyright (C) 2004  Davide Libenzi
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# Davide Libenzi <davidel@xmailserver.org>
#

use strict;

# Configuration variables
my $qlnx_qroot = $ENV{"QLNX_QROOT"};
my $qlnx_root = $ENV{"QLNX_ROOT"};
my $qlnx_repo = $ENV{"QLNX_REPO"};
my $qlnx_oinst;
my $qlnx_usehl = $ENV{"QLNX_USEHL"} || 0;
my $qlnx_cvsdirect = $ENV{"QLNX_CVSDIRECT"} || "--cvs-direct";
my $qlnx_bkcvs = $ENV{"QLNX_BKCVS"} || "--bkcvs";
my $qlnx_cfgfile = $ENV{"QLNX_CFGFILE"};
my $qlnx_outdir = $ENV{"QLNX_OUTDIR"} || ".";


my $cmdi;
my @cargs;


sub usage {
	print STDERR "use " . $0 . " [--qroot DIR] [--root DIR] [--repo NAME] [--outdir DIR]\n";
	print STDERR "\t[--cfgfile FILE] [--no-cvs-direct] [--no-bkcvs] [--hard-links] [--help]\n";
	print STDERR "\tcmd [arg [, arg]]\n\n";
	print STDERR "--qroot DIR        = Sets the tool root directory to DIR (mandatory)\n";
	print STDERR "--root DIR         = Sets the CVS root to DIR (mandatory)\n";
	print STDERR "--repo NAME        = Selects the repository inside the CVS root to NAME (mandatory)\n";
	print STDERR "--outdir DIR       = Sets the output directory to DIR (default .)\n";
	print STDERR "--cfgfile FILE     = Use the config file FILE to load options\n";
	print STDERR "--no-cvs-direct    = Unselect the CVS-direct mode of cvsps\n";
	print STDERR "--no-bkcvs         = Unselect the special --bkcvs mode of cvsps\n";
	print STDERR "--hard-links       = Use hard links to replicate caches\n";
	print STDERR "--help             = Prints this help\n";
	exit(2);
}

sub log_print {
	my ($lev, $msg) = @_;

	print STDERR "${msg}";
}

sub parse_opts {
	my (@opts) = @_;
	my $i;
	
	$cmdi = -1;
	for ($i = 0; $i <= $#opts; $i++) {
		if ($opts[$i] eq "--qroot") {
			if (++$i <= $#opts) {
				$qlnx_qroot = $opts[$i];
			}
		} elsif ($opts[$i] eq "--root") {
			if (++$i <= $#opts) {
				$qlnx_root = $opts[$i];
			}
		} elsif ($opts[$i] eq "--repo") {
			if (++$i <= $#opts) {
				$qlnx_repo = $opts[$i];
			}
		} elsif ($opts[$i] eq "--cfgfile") {
			if (++$i <= $#opts) {
				$qlnx_cfgfile = $opts[$i];
			}
		} elsif ($opts[$i] eq "--outdir") {
			if (++$i <= $#opts) {
				$qlnx_outdir = $opts[$i];
			}
		} elsif ($opts[$i] eq "--no-cvs-direct") {
			$qlnx_cvsdirect = "--no-cvs-direct";
		} elsif ($opts[$i] eq "--no-bkcvs") {
			$qlnx_bkcvs = "";
		} elsif ($opts[$i] eq "--hard-links") {
			$qlnx_usehl = 1;
		} elsif ($opts[$i] eq "--help") {
			usage();
		} else {
			$cmdi = $i;
			last;
		}
	}
}

sub bynuma { $a <=> $b }
sub bynumd { $b <=> $a }

sub load_config {
	my ($cfgfn) = @_;
	my @linez;

	if (!open(CFGF, "$cfgfn")) {
		print STDERR "unable to open: $cfgfn\n";
		return 0;
	}
	@linez = <CFGF>;
	close(CFGF);
	eval(join("", @linez));
	return 1;
}

sub patch_dump {
	my ($fn, $fd, $dlog) = @_;
	my $dump;

	if (!open(IFN, "$fn")) {
		print STDERR "unable to open: $fn\n";
		return 0;
	}
	$dump = $dlog;
	while (<IFN>) {
		if (!$dump && ($_ =~ /^Index: /)) {
			$dump = 1;
		}
		if ($dump) {
			print $fd $_;
		}
	}
	close(IFN);
	return 1;
}

sub file_append {
	my ($fh, $fname) = @_;
	my $data;
	my $size;
	
	if (!open(XFIL, "$fname")) {
		print STDERR "unable to open file: $fname\n";
		return 0;
	}
	binmode(XFIL);
	for (;;) {
		$size = read(XFIL, $data, 10000);
		if (!defined($size)) {
			print STDERR "unable to read file: $fname\n";
			close(XFIL);
			return 0;
		}
		if (!$size) {
			last;
		}
		if (!print $fh $data) {
			print STDERR "unable to write file\n";
			close(XFIL);
			return 0;
		}
	}
	close(XFIL);
	return 1;
}


sub file_copy {
	my ($dfname, $sfname) = @_;
	
	if (!open(QFIL, ">$dfname")) {
		print STDERR "unable to open file: $dfname\n";
		return 0;
	}
	binmode(QFIL);
	if (!file_append(*QFIL, $sfname)) {
		close(QFIL);
		unlink($dfname);
		return 0;
	}
	close(QFIL);
	return 1;
}

sub break_link {
	my ($fn) = @_;
	my $nlnk;
	my $tfn;

        $nlnk = (stat($fn))[3];
        if (!defined($nlnk) || $nlnk < 2) {
		return 0;
        }
	$tfn = $fn . ".__brklink__";
	if (!file_copy($tfn, $fn)) {
		return 0;
	}
	unlink($fn);
	if (!rename($tfn, $fn)) {
		return 0;
	}
	return 1;
}

sub links_break {
	my ($filez) = @_;
	my $i;

	for ($i = 0; $i <= $#$filez; $i++) {
		break_link($$filez[$i]);
	}
}

sub get_cache_pset {
	my ($cname, $psn) = @_;
	my %hm;
	my @ha;

	if ($cname =~ /^\#[0-9]+$/) {
		($$psn = $cname) =~ s/^\#([0-9]+).*/$1/;
		return 1;
	}
	if (!vmigs_get(\%hm, \@ha) || !defined($hm{$cname})) {
		return 0;
	}
	$$psn = $hm{$cname};
	return 1;
}

sub get_caches {
	my ($hc) = @_;
	my $i;
	my @clst;
	my @spth;
	my $cname;

	@clst = glob($qlnx_qroot . "/caches/*");
	for ($i = 0; $i <= $#clst; $i++) {
		@spth = split("/", $clst[$i]);
		$cname = $spth[$#spth];
		if (-d $clst[$i]) {
			$$hc{$cname} = $cname;
		}
	}
	return 1;
}

sub get_closet_cache {
	my ($psn, $hc) = @_;
	my @hk;
	my $i;
	my ($cc, $ccd);
	my $dst;

	@hk = keys(%$hc);
	$cc = -1;
	for ($i = 0; $i <= $#hk; $i++) {
		$dst = abs($psn - $hk[$i]);
		if (!defined($ccd) || $dst < $ccd) {
			$ccd = $dst;
			$cc = $hk[$i];
		}
	}
	return $cc;
}

sub load_psets {
	my ($fname, $hp, $top) = @_;

	if (!open(PSFIL, "$fname")) {
		print STDERR "unable to open ${fname}\n";
		return 0;
	}
	while (<PSFIL>) {
		chomp;
		if (/[0-9]+/ && ($top < 0 || $_ <= $top)) {
			$$hp{$_} = $_;
		}
	}
	close(PSFIL);
	return 1;
}

#
# Note: it modifies the hash passed in $hc
#
sub pset_diff {
	my ($hc, $hu, $hr) = @_;
	my $k;

	foreach $k (keys(%$hu)) {
		if (!defined($$hc{$k})) {
			$$hr{$k} = $k;
		} else {
			delete($$hc{$k});
		}
	}
	foreach $k (keys(%$hc)) {
		if (!defined($$hu{$k})) {
			$$hr{$k} = $k;
		}
	}
}

sub get_best_cache {
	my ($hp, $rbc, $rbch) = @_;
	my %hc;
	my $psfile;
	my ($bc, $bcc);
	my $c;

	if (!get_caches(\%hc)) {
		return 0;
	}
	$psfile = $qlnx_qroot . "/psets";
	foreach $c (keys(%hc)) {
		my $cpsn;
		my %hcs;
		my %hres;
		my $cc;

		if (!get_cache_pset($c, \$cpsn)) {
			return 0;
		}
		if (!load_psets($psfile, \%hcs, $cpsn)) {
			return 0;
		}
		pset_diff(\%hcs, $hp, \%hres);
		$cc = scalar(keys(%hres));
		if (!defined($bc) || $cc < $bcc) {
			$bc = $c . "\t" . $cpsn;
			$bcc = $cc;
			undef(%$rbch);
			%$rbch = %hres;
		}
	}
	$$rbc = $bc;
	return 1;
}

sub gen_psets {
	my ($fname, $hp) = @_;
	my $psfile;

	if (!open(PGFIL, "$fname")) {
		print STDERR "unable to open ${fname}\n";
		return 0;
	}
	$psfile = $qlnx_qroot . "/psets";
	while (<PGFIL>) {
		chomp;
		if (/^ALL$/) {
			undef(%$hp);
			if (!load_psets($psfile, $hp, -1)) {
				close(PGFIL);
				return 0;
			}
		} elsif (/^\@[a-zA-Z0-9\.\-_]+/) {
			my $ver;
			my %hm;
			my @ha;

			($ver = $_) =~ s/^\@([a-zA-Z0-9\.\-_]+).*/$1/;
			if (!vmigs_get(\%hm, \@ha)) {
				close(PGFIL);
				return 0;
			}
			if (!defined($hm{$ver})) {
				print STDERR "unknown version ${ver}\n";
				close(PGFIL);
				return 0;
			}
			undef(%$hp);
			if (!load_psets($psfile, $hp, $hm{$ver})) {
				close(PGFIL);
				return 0;
			}
		} elsif (/^%[0-9]+/) {
			my $psn;

			($psn = $_) =~ s/^%([0-9]+).*/$1/;
			undef(%$hp);
			if (!load_psets($psfile, $hp, $psn)) {
				close(PGFIL);
				return 0;
			}
		} elsif (/^\-[0-9]+/) {
			my $psnum;

			($psnum = $_) =~ s/^\-([0-9]+).*/$1/;
			if (defined($$hp{$psnum})) {
				delete($$hp{$psnum});
			}
		} elsif (/^\+[0-9]+/) {
			my $psnum;

			($psnum = $_) =~ s/^\+([0-9]+).*/$1/;
			$$hp{$psnum} = $psnum;
		} elsif (/^[\+\-][a-zA-Z0-9\.\-_]+\@[0-9\.]+$/) {
			$$hp{$_} = $_;
		} elsif (!/^[ \t]*#/ && !/^[ \t]*$/) {
			print STDERR "wrong syntax $_\n";
			close(PGFIL);
			return 0;
		}
	}
	close(PGFIL);
	return 1;
}

sub cache_install {
	my ($cname) = @_;
	my $hl;
	my $cdir;
	my $ddir;

	$cdir = $qlnx_qroot . "/caches/" . $cname;
	if (!-d $cdir) {
		print STDERR "missing cache directory ${cdir}\n";
		return 0;
	}
	$ddir = $qlnx_outdir . "/" . $qlnx_oinst;
	$hl = ($qlnx_usehl != 0) ? "-l": "";
	qx(cp -R $hl -f $cdir $ddir 2> /dev/null);
	if ($? != 0) {
		print STDERR "unable to instantiate cache to ${ddir}\n";
		return 0;
	}
	return 1;
}

sub get_mod_files {
	my ($pfname, $mfa) = @_;
	my $pln;
	my $mfile;
	my @spth;

	if (!open(PFIL, "$pfname")) {
		print STDERR "unable to open: $pfname\n";
		return 0;
	}
	$pln = "";
	while (<PFIL>) {
		chomp;
		if (($_ =~ /^\@\@ /) && ($pln =~ /^\+\+\+ /)) {
			($mfile = $pln) =~ s/\+\+\+ ([^ \t]+).*/$1/;
			@spth = split("/", $mfile);
			shift(@spth);
			$mfile = join("/", @spth);
			push(@$mfa, $mfile);
		}
		$pln = $_;
	}
	close(PFIL);
	return 1;
}

sub apply_patch {
	my ($bdir, $pfname, $rev, $fails) = @_;
	my @modf;
	my $i;
	my $ofile;

	if (!get_mod_files($pfname, \@modf)) {
		return 0;
	}
	if ($#modf < 0) {
		return 1;
	}
	for ($i = 0; $i <= $#modf; $i++) {
		$ofile = $bdir . "/" . $modf[$i] . ".orig";
		if (-e $ofile) {
			unlink($ofile);
		}
		$ofile = $bdir . "/" . $modf[$i] . ".rej";
		if (-e $ofile) {
			unlink($ofile);
		}
	}
	if ($qlnx_usehl) {
		links_break(\@modf);
	}
	$rev = ($rev != 0) ? "-R": "";
	qx(patch -d $bdir -f -p1 $rev < $pfname 2> /dev/null);
	if ($? != 0) {
		for ($i = 0; $i <= $#modf; $i++) {
			$ofile = $bdir . "/" . $modf[$i] . ".rej";
			if (-e $ofile) {
				push(@$fails, $modf[$i]);
			}
		}
		return 2;
	}
	for ($i = 0; $i <= $#modf; $i++) {
		$ofile = $bdir . "/" . $modf[$i] . ".orig";
		unlink($ofile);
	}
	return 1;
}

sub vmig_ext {
	my ($ifd, $ofd, $ca) = @_;
	my $cpset = "0";
	my $cver;
	
	while (<$ifd>) {
		chomp;
		if (/^PatchSet [0-9]+/) {
			($cpset = $_) =~ s/PatchSet ([0-9]+).*/$1/;
			push(@$ca, $cpset);
		}
		if (/^v\d+\.\d+(\.\d+)* -> v\d+\.\d+(\.\d+)*/) {
			($cver = $_) =~ s/.* -> v([0-9\.]+).*/$1/;
			print $ofd "${cver}\t${cpset}\n";
		}
		if (/^Tag: v\d+_\d+_\d+.*/) {
			($cver = $_) =~ s/Tag: v(.*)/$1/;
			$cver =~ tr/_/\./;
			print $ofd "${cver}\t${cpset}\n";
		}
	}
	return $cpset;
}

sub update_cache {
	my $logfile;

	$logfile = $qlnx_qroot . "/logfile";
	qx(cvsps -u --root $qlnx_root $qlnx_bkcvs $qlnx_cvsdirect -q $qlnx_repo > $logfile 2> /dev/null);
        if ($? != 0) {
		return 0;
        }
	return 1;
}

sub gen_missing_psets {
	my ($from, $to) = @_;
	my $psdir;

	$psdir = $qlnx_qroot . "/patches";
	qx(cvsps --root $qlnx_root $qlnx_cvsdirect -g -s $from-$to -p $psdir -q $qlnx_repo 2> /dev/null);
        if ($? != 0) {
		return 0;
        }
	return 1;
}

sub update_local {
	my ($logfile, $migfile, $psfile, $cafile);
	my ($lca, $cca);
	my @ca;

	$logfile = $qlnx_qroot . "/logfile";
	if (!open(LFIL, "$logfile")) {
		print STDERR "unable to open ${logfile}\n";
		return 0;
	}
	$migfile = $qlnx_qroot . "/migfile";
	if (!open(MFIL, ">$migfile")) {
		print STDERR "unable to create ${migfile}\n";
		close(LFIL);
		return 0;
	}
	$cca = vmig_ext(*LFIL, *MFIL, \@ca);
	close(MFIL);
	close(LFIL);
	$psfile = $qlnx_qroot . "/psets";
	if (!open(PFIL, ">$psfile")) {
		print STDERR "unable to create ${psfile}\n";
		return 0;
	}
	print PFIL join("\n", @ca, "");
	close(PFIL);
	$lca = 0;
	$cafile = $qlnx_qroot . "/lastps";
	if (open(CFIL, "$cafile")) {
		$lca = <CFIL>;
		close(CFIL);
		chomp($lca);
	}
	if (!gen_missing_psets($lca + 1, $cca)) {
		return 0;
	}
	if (!open(CFIL, ">$cafile")) {
		print STDERR "unable to create ${cafile}\n";
		return 0;
	}
	print CFIL "${cca}\n";
	close(CFIL);
	return 1;
}

sub gen_psets_range {
	my ($from, $to) = @_;
	my $cafile;
	my $lca;

	$lca = 0;
	$cafile = $qlnx_qroot . "/lastps";
	if (open(CFIL, "$cafile")) {
		$lca = <CFIL>;
		close(CFIL);
		chomp($lca);
	}
	if (!gen_missing_psets($from, $to)) {
		return 0;
	}
	if ($to > $lca) {
		if (!open(CFIL, ">$cafile")) {
			print STDERR "unable to create ${cafile}\n";
			return 0;
		}
		print CFIL "${to}\n";
		close(CFIL);
	}
	return 1;
}

sub vmigs_get {
	my ($hm, $ha) = @_;
	my $migfile;
	my @aasc;

	$migfile = $qlnx_qroot . "/migfile";
	if (!open(MFIL, "$migfile")) {
		print STDERR "unable to open ${migfile}\n";
		return 0;
	}
	while (<MFIL>) {
		chomp;
		@aasc = split(/[ \t]+/, $_);
		if ($#aasc > 0) {
			push(@$ha, $aasc[0]);
			$$hm{$aasc[0]} = $aasc[1];
		}
	}
	close(MFIL);
	return 1;
}

sub get_logs {
	my ($hl) = @_;
	my $logfile;
	my $nkeys;
	my $ln;
	my ($cps, $ocps);
	my @lga;

	$logfile = $qlnx_qroot . "/logfile";
	if (!open(LFIL, "$logfile")) {
		print STDERR "unable to open ${logfile}\n";
		return 0;
	}
	$nkeys = scalar(keys(%$hl));
	while (<LFIL>) {
		$ln = $_;
		chomp($ln);
		if ($ln =~ /^PatchSet [0-9]+/) {
			($cps = $ln) =~ s/PatchSet ([0-9]+).*/$1/;
			if (defined($ocps) && defined($$hl{$ocps})) {
				pop(@lga);
				$$hl{$ocps} = join("", @lga);
				$nkeys--;
				if ($nkeys == 0) {
					last;
				}
			}
			$ocps = $cps;
			undef(@lga);
		}
		push(@lga, $ln . "\n");
	}
	if (defined($ocps) && defined($$hl{$ocps})) {
		pop(@lga);
		$$hl{$ocps} = join("", @lga);
	}
	close(LFIL);
	return 1;
}

sub search_logs {
	my ($rex, $hf) = @_;
	my $logfile;
	my $ln;
	my ($cps, $ocps);
	my $logd;
	my @lga;

	$logfile = $qlnx_qroot . "/logfile";
	if (!open(LFIL, "$logfile")) {
		print STDERR "unable to open ${logfile}\n";
		return 0;
	}
	while (<LFIL>) {
		$ln = $_;
		chomp($ln);
		if ($ln =~ /^PatchSet [0-9]+/) {
			($cps = $ln) =~ s/PatchSet ([0-9]+).*/$1/;
			if (defined($ocps)) {
				pop(@lga);
				$logd = join("", @lga);
				if ($logd =~ /$rex/) {
					$$hf{$ocps} = $logd;
				}
			}
			$ocps = $cps;
			undef($logd);
			undef(@lga);
		}
		push(@lga, $ln . "\n");
	}
	if (defined($ocps)) {
		pop(@lga);
		$logd = join("", @lga);
		if ($logd =~ /$rex/) {
			$$hf{$ocps} = $logd;
		}
	}
	close(LFIL);
	return 1;
}

sub prepare_journal {
	my ($cfgfn, $jfn) = @_;
	my %hps;
	my $bc;
	my %bch;
	my @bcx;
	my $cs;

	if (!gen_psets($cfgfn, \%hps)) {
		return 0;
	}
	if (!get_best_cache(\%hps, \$bc, \%bch) || !defined($bc)) {
		print STDERR "no cache available\n";
		return 0;
	}
	@bcx = split(/\t/, $bc);
	if (!open(JFIL, ">$jfn")) {
		print STDERR "unable to create file ${jfn}";
		return 0;
	}
	print JFIL "*" . $bcx[0] . "\n";
	$bc = $bcx[1];
	foreach $cs (sort {
		my ($aa, $bb);
		$aa = $a;
		$bb = $b;
		if ($aa =~ /\@/) {
			my @sa;

			@sa = split(/\@/, $aa);
			$aa = $sa[1];
		}
		if ($bb =~ /\@/) {
			my @sa;

			@sa = split(/\@/, $bb);
			$bb = $sa[1];
		}
		abs($aa - $bc) <=> abs($bb - $bc);
	} keys(%bch)) {
		if ($cs =~ /\@/) {
			my @sa;

			@sa = split(/\@/, $cs);
			print JFIL "|" . $sa[0] . "\n";
		} elsif ($cs > $bc) {
			print JFIL "+${cs}\n";
		} else {
			print JFIL "-${cs}\n";
		}
	}
	close(JFIL);
	return 1;
}

sub start_checkout {
	my ($cfgfn) = @_;
	my ($jfn);

	$jfn = $qlnx_outdir . "/journal";
	if (!prepare_journal($cfgfn, $jfn)) {
		return 0;
	}

	return 1;
}

sub flush_journal {
	my ($jfn, $jops) = @_;

	if (!open(JFIL, ">$jfn")) {
		print STDERR "unable to create file ${jfn}";
		return 0;
	}
	print JFIL join("", @$jops);
	close(JFIL);
	return 1;
}

sub add_patch {
	my ($cs) = @_;
	my $rev;
	my $ddir;
	my $pfname;
	my @fails;
	my $pres;

	if ($cs !~ /^[\+\-][a-zA-Z0-9\.\-_]+/) {
		print STDERR "wrong cset format ${cs}\n";
		return 0;
	}
	$ddir = $qlnx_outdir . "/" . $qlnx_oinst;
	$rev = ($cs =~ /^\-/) ? 1: 0;
	$cs =~ s/^[\+\-]([a-zA-Z0-9\.\-_]+).*/$1/;
	$pfname = $qlnx_qroot . "/patches/" . $cs . ".patch";
	if (!-e $pfname) {
		$pfname = $qlnx_qroot . "/xpatches/" . $cs;
		if (!-e $pfname) {
			print STDERR "unable to find patch ${cs}\n";
			return 0;
		}
	}
	log_print(5, ($rev ? "removing": "adding") . " patch ${cs} ...\n");
	$pres = apply_patch($ddir, $pfname, $rev, \@fails);
	if (!$pres) {
		return 0;
	}
	if ($pres > 1) {
		log_print(1, "********************************\n");
		log_print(1, "failed to " . ($rev ? "remove": "add") . " patch ${cs}\n");
		log_print(1, "files with rejections:\n" . join("\n", @fails, ""));
		return 0;
	}
	return 1;
}

sub resume_checkout {
	my $jfn;
	my @jops;
	my $cs;
	my $ddir;

	$jfn = $qlnx_outdir . "/journal";
	if (!open(JFIL, "$jfn")) {
		print STDERR "unable to open file ${jfn}";
		return 0;
	}
	@jops = <JFIL>;
	close(JFIL);
	$ddir = $qlnx_outdir . "/" . $qlnx_oinst;
	while (($cs = shift(@jops))) {
		chomp($cs);
		if ($cs =~ /^\*.+/) {
			$cs =~ s/\*(.+)/$1/;
			log_print(5, "installing cache ${cs} ...\n");
			if (!cache_install($cs)) {
				return 0;
			}
			if (!flush_journal($jfn, \@jops)) {
				return 0;
			}
		} elsif ($cs =~ /^[\+\-][0-9]+/) {
			if (!flush_journal($jfn, \@jops)) {
				return 0;
			}
			if (!add_patch($cs)) {
				return 0;
			}
		} elsif ($cs =~ /^\|[\+\-][a-zA-Z0-9\.\-_]+/) {
			if (!flush_journal($jfn, \@jops)) {
				return 0;
			}
			$cs =~ s/^\|(.*)/$1/;
			if (!add_patch($cs)) {
				return 0;
			}
		}
	}
	return 1;
}

sub check_qroot {
	my ($qr) = @_;
	my $qsr;

	if (!(-d $qr) && !mkdir($qr, 0775)) {
		print STDERR "unable to create directory ${qr}";
		return 0;
	}
	$qsr = $qr . "/patches";
	if (!(-d $qsr) && !mkdir($qsr, 0775)) {
		print STDERR "unable to create directory ${qsr}";
		return 0;
	}
	$qsr = $qr . "/xpatches";
	if (!(-d $qsr) && !mkdir($qsr, 0775)) {
		print STDERR "unable to create directory ${qsr}";
		return 0;
	}
	$qsr = $qr . "/caches";
	if (!(-d $qsr) && !mkdir($qsr, 0775)) {
		print STDERR "unable to create directory ${qsr}";
		return 0;
	}

	return 1;
}


#
# All starts here ... (and ends pretty soon)
#

parse_opts(@ARGV);
if (defined($qlnx_cfgfile) && !load_config($qlnx_cfgfile)) {
	exit(1);
}
if (!defined($qlnx_qroot)) {
	print STDERR "undefined qlnx root directory\n";
	usage();
}
$qlnx_qroot =~ s/(.*)\/$/$1/;
if (!defined($qlnx_root)) {
	print STDERR "undefined CVS root directory\n";
	usage();
}
$qlnx_root =~ s/(.*)\/$/$1/;
if (!defined($qlnx_repo)) {
	print STDERR "undefined CVS repository\n";
	usage();
}
if (!defined($qlnx_oinst)) {
	$qlnx_oinst = $qlnx_repo;
}
if ($cmdi < 0) {
	print STDERR "command not specified\n";
	usage();
}
if (!check_qroot($qlnx_qroot)) {
	exit(1);
}
$qlnx_outdir =~ s/(.*)\/$/$1/;

@cargs = @ARGV;
splice(@cargs, 0, $cmdi);

if (($cargs[0] =~ /^cupdate$/) || ($cargs[0] =~ /^cu$/)) {
	if (!update_cache() || !update_local()) {
		exit(1);
	}
} elsif (($cargs[0] =~ /^migs/) || ($cargs[0] =~ /^mg/)) {
	my %hm;
	my @ha;
	my $i;

	if (!vmigs_get(\%hm, \@ha)) {
		exit(1);
	}
	for ($i = 0; $i <= $#ha; $i++) {
		my ($j, $emit);

		$emit = 0;
		for ($j = 1; $j <= $#cargs; $j++) {
			if ($ha[$i] =~ /^$cargs[$j]$/) {
				$emit++;
				last;
			}
		}
		if ($emit || $#cargs < 1) {
			print $ha[$i] . "\t" . $hm{$ha[$i]} . "\n";
		}
	}
} elsif (($cargs[0] =~ /^logs$/) || ($cargs[0] =~ /^lg$/)) {
	my %hl;
	my $i;

	for ($i = 1; $i <= $#cargs; $i++) {
		$hl{$cargs[$i]} = "";
	}
	if (!get_logs(\%hl)) {
		exit(1);
	}
	for ($i = 1; $i <= $#cargs; $i++) {
		print "********************************\n";
		if ($hl{$cargs[$i]} eq "") {
			print ">>> PatchSet " . $cargs[$i] . " not found\n";
		} else {
			print $hl{$cargs[$i]};
		}
	}
} elsif (($cargs[0] =~ /^getpatch$/) || ($cargs[0] =~ /^gp$/)) {
	my $i;
	my $psname;

	for ($i = 1; $i <= $#cargs; $i++) {
		$psname = $qlnx_qroot . "/patches/" . $cargs[$i] . ".patch";
		print "********************************\n";
		if (-e $psname) {
			print "PatchSet " . $cargs[$i] . "\n";
			patch_dump($psname, *STDOUT, 0);
		} else {
			print ">>> PatchSet " . $cargs[$i] . " not found\n";
		}
	}
} elsif (($cargs[0] =~ /^searchlogs$/) || ($cargs[0] =~ /^sl$/)) {
	my %hf;
	my @ok;
	my $i;

	if ($#cargs < 1) {
		print STDERR "search pattern missing\n";
		exit(1);
	}
	if (!search_logs($cargs[1], \%hf)) {
		exit(1);
	}
	@ok = sort bynuma keys(%hf);
	for ($i = 0; $i <= $#ok; $i++) {
		print "********************************\n";
		print $hf{$ok[$i]};
	}
} elsif (($cargs[0] =~ /^ckstart$/) || ($cargs[0] =~ /^cs$/)) {
	if ($#cargs < 1) {
		print STDERR "search pattern missing\n";
		exit(1);
	}
	if (!start_checkout($cargs[1]) || !resume_checkout()) {
		exit(1);
	}
} elsif (($cargs[0] =~ /^genpsets$/) || ($cargs[0] =~ /^ge$/)) {
	if ($#cargs < 2) {
		print STDERR "pset range missing\n";
		exit(1);
	}
	if (!gen_psets_range($cargs[1], $cargs[2])) {
		exit(1);
	}
} elsif (($cargs[0] =~ /^ckresume$/) || ($cargs[0] =~ /^cr$/)) {
	if (!resume_checkout()) {
		exit(1);
	}
} elsif (($cargs[0] =~ /^patch$/) || ($cargs[0] =~ /^pa$/)) {
	my $i;

	for ($i = 1; $i <= $#cargs; $i++) {
		if (!add_patch($cargs[$i])) {
			exit(1);
		}
	}
}

exit(0);

